/*****************************************************************************
 * prodos/misc.c
 * Miscellaneous utility and helper functions.
 *
 * Apple II ProDOS Filesystem Driver for Linux 2.4.x
 * Copyright (c) 2001 Matt Jensen.
 * This program is free software distributed under the terms of the GPL.
 *
 * 18-May-2001: Created
 *****************************************************************************/

/*= Kernel Includes =========================================================*/

#include <linux/blkdev.h>

/*= ProDOS Includes =========================================================*/

#include "prodos.h"

/*= Private Functions =======================================================*/

/*= Exported Functions ======================================================*/

/*****************************************************************************
 * prodos_check_name()
 * Check a name for validity under ProDOS naming rules.
 *****************************************************************************/
int prodos_check_name(struct qstr *qstr) {
	int len = qstr->len;
	char c = 0;

	/* Disregard any fork suffix. */
	if (qstr->name[len - 2] == '_') {
		c = PRODOS_TOLOWER(qstr->name[len - 1]);
		if (c == 'm' || c == 'r') len -= 2;
	}

	/* Check only up to ProDOS name length limit. */
	len = PRODOS_MIN(len,PRODOS_MAX_NAME_LEN);

	/* Is the name made up of valid characters? */
	while (len > 0) {
		len--;
		c = PRODOS_TOUPPER(qstr->name[len]);
		if (!len) {
			if (c < 'A' || c > 'Z') return -EINVAL;
		}
		else {
			if (!((c >= 'A' && c <= 'Z') ||
			    (c >= '0' && c <= '9') ||
				(c == '.'))) return -EINVAL;
		}
	}
	return 0;
}

/*****************************************************************************
 * prodos_bread()
 * Read a block from a device.
 *****************************************************************************/
struct buffer_head *prodos_bread(struct super_block *sb,int block_num) {
	SHORTCUT struct prodos_sb_info * const psb = PRODOS_SB(sb);
	if (block_num < 0 || block_num >= psb->s_part_size) return NULL;
	return bread(sb->s_dev,psb->s_part_start + block_num,PRODOS_BLOCK_SIZE);
}

/*****************************************************************************
 * prodos_brelse()
 * Release a block that was previously read via prodos_bread().
 *****************************************************************************/
void prodos_brelse(struct buffer_head *bh) {
	brelse(bh);
}


